using System;
using System.Collections;
using System.Xml;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// This class provides utility methods for XML parsing. It performs necessary checks 
	/// while retrieving XML nodes and throws XmlParseException if needed.
	/// It must be used for VistALink objects serialization / deserialization / parsing only.
	/// </summary>
	public sealed class XmlUtility
	{
		/// <summary>
		/// This class shouldn't be initialized. It only provides a number of static utility methods
		/// helping with XML parsing / deserialization.
		/// </summary>
		private XmlUtility() {}

		/// <summary>
		/// Writes out element with child CDATA node containing a given value.
		/// </summary>
		/// <param name="writer">XmlWriter to use.</param>
		/// <param name="elementName">Name of element to create.</param>
		/// <param name="cdataContents">Value to write to CDATA.</param>
		public static void GenWriteElementWithCData( XmlWriter writer, string elementName, string cdataContents )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			if( elementName == null )
				throw( new ArgumentNullException( "elementName" ) );

			writer.WriteStartElement( elementName );
			writer.WriteCData( cdataContents );
			writer.WriteEndElement();
		}

		/// <summary>
		/// Verifies that given XML element has an attribute with a specified name. 
		/// If it does not - XmlParseException is thrown. 
		/// </summary>
		/// <param name="element">XML element to check.</param>
		/// <param name="attributeName">Name of attribute to verify.</param>
		public static void ParseCheckRequiredAttribute( XmlElement element, string attributeName )
		{
			if( element == null )
				throw( new ArgumentNullException( "element" ) );

			if( attributeName == null )
				throw( new ArgumentNullException( "attributeName" ) );

			if( !element.HasAttribute( attributeName ) )
				throw( new XmlParseException( 
					SR.Exceptions.XmlParseRequiredAttributeMissing( element.Name, attributeName ) ) );
		}

		/// <summary>
		/// Verifies that an attribute with a given name for a specified XML element holds an expected value.
		/// If it does not - XmlParseException is thrown.
		/// </summary>
		/// <param name="element">XML element to check.</param>
		/// <param name="attributeName">Name of attribute to verify.</param>
		/// <param name="expectedValue">Expected attribute value.</param>
		private static void ParseCheckAttributeValue( XmlElement element, string attributeName, string expectedValue ) 
		{
			if( element == null )
				throw( new ArgumentNullException( "element" ) );

			if( attributeName == null )
				throw( new ArgumentNullException( "attributeName" ) );			

			if( element.GetAttribute( attributeName ) != expectedValue )
				throw( new XmlParseException(
					SR.Exceptions.XmlParseCheckAttributeValueMismatch( attributeName, element.Name, element.GetAttribute( attributeName ), expectedValue ) ) );
		}

		/// <summary>
		/// Verifies that a given XML element contains specified attribute and
		/// returns this attribute value. 
		/// </summary>
		/// <param name="element">XML element to work with.</param>
		/// <param name="attributeName">Attribute name.</param>
		/// <returns>Attribute value.</returns>
		public static string ParseGetRequiredAttributeValue( XmlElement element, string attributeName )
		{
			ParseCheckRequiredAttribute( element, attributeName );
			return element.GetAttribute( attributeName );
		}

		/// <summary>
		/// Verifies that given XML element has an attribute with a specified name and that
		/// this attribute holds an expected value.
		/// If it does not - XmlParseException is thrown.
		/// </summary>
		/// <param name="element">XML element to check.</param>
		/// <param name="attributeName">Name of attribute to verify.</param>
		/// <param name="expectedValue">Expected attribute value.</param>
		public static void ParseCheckRequiredAttributeValue( XmlElement element, string attributeName, string expectedValue )
		{
			ParseCheckRequiredAttribute( element, attributeName );
			ParseCheckAttributeValue( element, attributeName, expectedValue );
		}

		/// <summary>
		/// Verifies that one and only one element with specified name exists under the root node, 
		/// has child CDATA node and returns content of CDATA node.
		/// </summary>
		/// <param name="root">Root element.</param>
		/// <param name="elementTagName">Element tag name.</param>
		/// <returns>Value contained in CDATA.</returns>
		public static string ParseGetRequiredElementWithUniqueNameCDataValue( XmlElement root, string elementTagName )
		{
			XmlElement _cdataRoot = ParseGetRequiredElementByUniqueTagName( root, elementTagName );
			
			if( _cdataRoot.FirstChild == null || _cdataRoot.FirstChild.NodeType != XmlNodeType.CDATA )
				throw( new XmlParseException( SR.Exceptions.XmlParseCDataNotFoundWhereExpected( elementTagName ) ) );

			return _cdataRoot.FirstChild.Value;
		}

		/// <summary>
		/// Verifies that one and only one element with a given name exists 
		/// in a specified XML document and returns reference to it. 
		/// </summary>
		/// <param name="sourceDoc">XML document to use.</param>
		/// <param name="elementTagName">Element tag name.</param>
		/// <returns>Reference to verified element.</returns>
		public static XmlElement ParseGetRequiredElementByUniqueTagName( XmlDocument sourceDoc, string elementTagName )
		{
			XmlElement _result = ParseGetOptionalElementByUniqueTagName( sourceDoc, elementTagName );

			if( _result == null )
				throw( new XmlParseException( SR.Exceptions.XmlParseRequiredElementNotFound( elementTagName ) ) );

			return _result;
		}

		/// <summary>
		/// Verifies that one and only one element with a given name exists under a 
		/// specified root element and returns reference to it. 
		/// </summary>
		/// <param name="root">Root XML element to use.</param>
		/// <param name="elementTagName">XML element tag name.</param>
		/// <returns>Reference to verified XML element.</returns>
		public static XmlElement ParseGetRequiredElementByUniqueTagName( XmlElement root, string elementTagName )
		{
			XmlElement _result = ParseGetOptionalElementByUniqueTagName( root, elementTagName );

			if( _result == null )
				throw( new XmlParseException( SR.Exceptions.XmlParseRequiredElementNotFound( elementTagName ) ) );

			return _result;
		}

		/// <summary>
		/// Verifies that no more than one element with a specified tag name exists
		/// in a given XML document and returns a reference to element when exactly one
		/// element was found. Otherwise returns null.
		/// </summary>
		/// <param name="sourceDoc">XML document to use.</param>
		/// <param name="elementTagName">XML element tag name.</param>
		/// <returns>Reference to found XML element, null if element was not found.</returns>
		public static XmlElement ParseGetOptionalElementByUniqueTagName( XmlDocument sourceDoc, string elementTagName )
		{
			if( sourceDoc == null ) 
				throw( new ArgumentNullException( "sourceDoc" ) );

			if( sourceDoc.DocumentElement == null )
				return null;

			return ParseGetOptionalElementByUniqueTagName( sourceDoc.DocumentElement, elementTagName );
		}

		/// <summary>
		/// Verifies that no more than one element with a specified tag name exists
		/// under a given XML root node and returns a reference to element when exactly one
		/// element was found. Otherwise returns null.
		/// </summary>
		/// <param name="root">Root XML element to use.</param>
		/// <param name="elementTagName">XML element tag name.</param>
		/// <returns>Reference to found XML element, null if element was not found.</returns>
		public static XmlElement ParseGetOptionalElementByUniqueTagName( XmlElement root, string elementTagName )
		{
			if( root == null )
				throw( new ArgumentNullException( "root" ) );

			if( elementTagName == null )
				throw( new ArgumentNullException( "elementTagName" ) );

			XmlNodeList _list = root.GetElementsByTagName( elementTagName );

			if( _list.Count == 0 )
				return null;

			if( _list.Count > 1 ) 
				throw( new XmlParseException( SR.Exceptions.XmlParseMoreThanOneElementWasFound( elementTagName, root.Name ) ) );

			return (XmlElement)_list[0];
		}

		/// <summary>
		/// This method accepts an XML node, verifies that this node is XML element with a specified name
		/// and returns reference to the same node cast to XmlElement.
		/// </summary>
		/// <param name="node">Source XML node to work with.</param>
		/// <param name="elementTagName">XmlElement tag name to ensure.</param>
		/// <returns>Reference to source XmlNode cast as XmlElement.</returns>
		public static XmlElement ParseValidateConvertNodeToElement( XmlNode node, string elementTagName )
		{
			if( node == null )
				throw( new ArgumentNullException( "node" ) );

			if( elementTagName == null )
				throw( new ArgumentNullException( "ElementTagName" ) );

			if( !(node is XmlElement) || ((XmlElement)node).Name != elementTagName )
				throw( new XmlParseException( SR.Exceptions.XmlParseRequiredElementNotFound( elementTagName ) ) );

			return (XmlElement)node;
		}

		/// <summary>
		/// Custom implementation of <see cref="XmlNodeList"/> allowing creating methods 
		/// that return list of nodes in <see cref="XmlNodeList"/>.
		/// </summary>
		private class XmlNodeListAccessor : XmlNodeList
		{
			private ArrayList _nodes;

			public XmlNodeListAccessor() : base() 
			{
				_nodes = new ArrayList();
			}

			public override XmlNode Item( int index )
			{
				return (XmlNode)_nodes[ index ];
			}
			
			public override IEnumerator GetEnumerator()
			{
				return _nodes.GetEnumerator();
			}

			public override int Count
			{
				get
				{
					return _nodes.Count;
				}
			}

			public void Add( XmlNode node )
			{
				if( !_nodes.Contains( node ) )
					_nodes.Add( node );
			}
		}

		/// <summary>
		/// Returns list of child elements for a given XML node. 
		/// Note that although method returns <see cref="XmlNodeList"/>, changes made
		/// to structure of underlying <see cref="XmlDocument"/> will not be reflected in 
		/// returned <see cref="XmlNodeList"/>.
		/// </summary>
		/// <param name="node"><see cref="XmlNode"/> to extract child elements from.</param>
		/// <returns>List of child elements.</returns>
		public static XmlNodeList ParseGetChildElements( XmlNode node )
		{
			if( node == null )
				throw( new ArgumentNullException( "node" ) );

			XmlNodeListAccessor _list = new XmlNodeListAccessor();

			foreach( XmlNode _childNode in node.ChildNodes )
				if( _childNode.NodeType == XmlNodeType.Element )
					_list.Add( _childNode );

			return (XmlNodeList)_list;
		}
	}
}
